
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Mon Mar 28 12:09:38 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_SST_TST_ATP_Sym.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ***********************************/
#include "SST_TST.h"
#if TST_USE_CRYS_SST_SYM_MODULE
/************************ Defines ******************************/
DxUint8_t            TST_InputData[TST_VECTOR_SIZE] = {0};
DxUint8_t            TST_EncryptedData[TST_VECTOR_SIZE] = {0};
DxUint8_t            TST_DecryptedData[TST_VECTOR_SIZE] = {0};
CRYS_AES_IvCounter_t TST_IVCounter = {0};

/************************ Enums ********************************/
/************************ Typedefs *****************************/
/************************ Global Data **************************/
/************* Private function prototype **********************/
/************************ Public Functions *********************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_SST_TST_ATP_AES
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 *    Test the connection between CRYS and the SSTfunctionality while using SST AES generated keys
 *    in the CRYS_SST module
 *  
 * Algorithm:
 *    1.    Create a new session by calling: SST_SessionCreate.
 *    2.    Open a transaction by calling: SST_TransactionStart.
 *    3.    Create a new master password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    4.    Close the transaction by calling: SST_TransactionEnd
 *    5.    Open the  password authentication by calling: SST_AuthenticationOpen.
 *    6.    Open a transaction by calling: SST_TransactionStart.
 *    7.    Generate AES keys inside the SST by calling: CRYS_SST_KG_AES.
 *    8.    Read the Key from the SST by calling: SST_AESKeyRead.
 *    9.    Encrypt the data using CRYS and extracted data(key) by calling = CRYS_AES.
 *    10.   Decrypt the data using CRYS_SST by calling: CRYS_SST_AES.
 *    11.   Compare the decrypted result to the original
 *    12.   Decrypt the data using the CRYS_SST non-integrated function
 *    13.   Compare the decrypted result to the original
 *    14.   Delete the AES key by calling: SST_DataDelete.
 *    15.   Close the authenticator by calling: SST_AuthenticationClose.
 *    16.   Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    17.   Close the transaction by calling: SST_TransactionClose.
 *    18.   Delete the session by calling: SST_SessionDelete.
 *                                
 *******************************************************************************/    
DxError_t CRYS_SST_TST_ATP_AES ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_SST_TST_ATP_AES";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_AES_KeyHandle,TST_InitialCtrlAuth,TST_PasswordAuthHandle;
    DxUint8_t                   TST_AES_KeySizes;
    CRYS_AES_KeySize_t          TST_AES_KeySize = CRYS_AES_Key128BitSize;    
    CRYS_AES_KeySize_t          TST_OutputKeySizeBits = CRYS_AES_Key128BitSize;
    DxUint8_t                   TST_OutputKey[32];
    SSTAESKeyType_t             TST_SSTAESKeyType;
    CRYS_AESUserContext_t       TST_AESCtx;
    DxError_t                   TST_Status;

    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** CRYS_SST_TST_ATP_AES Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        if (TST_HandleType == TST_INTERNAL_HANDLE)
        {   
            ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
            ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
            ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
            TST_AES_KeyHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            TST_PasswordAuthHandle.objId =  SST_HANDLE_RANDOM_GENERATE;
           
        }
        else
        {
            ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
            ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
            ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
            TST_AES_KeyHandle.objId = TST_TEST_DEFINED_HENDLE;
            TST_PasswordAuthHandle.objId =  TST_TEST_DEFINED_HENDLE;
            
        }  


       /* Create a session for operations */
       /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                                  "CRYS_SST_TST_ATP_AES",
                                  TST_MasterTestName);


      /* Open a transaction for operations */
      /*-----------------------------------*/
      TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                                "CRYS_SST_TST_ATP_AES",
                                 TST_MasterTestName);


      /* Create a new  password authenticator */
      /*--------------------------------------*/
      TST_InitialCtrlAuth.objDigest = 0;
      TST_InitialCtrlAuth.objId = SST_CLOSE_AUTH_HANDLE;
      TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                   TST_SessionId, 
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterPassword,
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterPasswordSizeInBytes,
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterMaxNumOfTrials,
                                                   TST_InitialCtrlAuth,
                                                   &TST_PasswordAuthHandle,
                                                   TST_Workspace,
                                                   TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate ",
                                 CRYS_SST_TST_ATP_AES,
                                 TST_MasterTestName);


      /* Close the transaction */
      /*-----------------------*/
      TST_Status = SST_TransactionEnd(TST_TransactionId, DX_TRUE, TST_Workspace, TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                                 CRYS_SST_TST_ATP_AES,
                                 TST_MasterTestName);


      /* Open the  password authentication */
      /*-----------------------------------------*/
      TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                          TST_PasswordAuthHandle,
                                          TST_ATP_PassAuthObjectTestData[0].TST_MasterPassword,
                                          TST_ATP_PassAuthObjectTestData[0].TST_MasterPasswordSizeInBytes,
                                          TST_NO_CHALLENGE,
                                          TST_Workspace,
                                          TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen ",
                                 CRYS_SST_TST_ATP_AES,
                                  TST_MasterTestName);


      /* Open a transaction for operations */
      /*-----------------------------------*/
      TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                                 CRYS_SST_TST_ATP_AES,
                                 TST_MasterTestName);

      /* Setting test case specific parameters for key sizes */
      /*To run on all key types vectors*/
     for (TST_AES_KeySizes =0 ; TST_AES_KeySizes < TST_NUM_OF_AES_KEYS ; TST_AES_KeySizes++)
     {
         switch(TST_AES_KeySizes)
         {
     
          case 0:
                 TST_AES_KeySize = CRYS_AES_Key128BitSize;
                 TST_OutputKeySizeBits = CRYS_AES_Key128BitSize;
                 TST_SSTAESKeyType = SST_KEY_TYPE_AES_128_BIT;
                 ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** AES Key 128 bit\n"));
                 break;
          case 1:
                 if (TST_SST_USE_AES_192_KEY == 0) 
                     goto END_OF_TEST; 
                 else
                 {
                     TST_AES_KeySize = CRYS_AES_Key192BitSize;
                     TST_OutputKeySizeBits = CRYS_AES_Key192BitSize;
                     TST_SSTAESKeyType = SST_KEY_TYPE_AES_192_BIT;
                     ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** AES Key 192 bit\n"));    
                 }
                 break; 
          case 2:
                 if (TST_SST_USE_AES_256_KEY == 0) 
                     goto END_OF_TEST;
                 else 
                 {
                     TST_AES_KeySize = CRYS_AES_Key256BitSize;
                     TST_OutputKeySizeBits = CRYS_AES_Key256BitSize;
                     TST_SSTAESKeyType = SST_KEY_TYPE_AES_256_BIT;
                     ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** AES Key 256 bit\n"));    
                 }                            
                 break; 
          default:
                 ATP_LOG_TST_PRINT((SST_TST_MSG,"\n***  wrong AES Key \n"));
        };

        /* Generate AES keys inside the SST */
        /*----------------------------------*/
        TST_Status = CRYS_SST_KG_AES(TST_TransactionId,
                                     TST_SessionId,
                                     TST_PasswordAuthHandle,
                                     TST_AES_KEY_TYPE,
                                     TST_AES_KeySize,
                                     &TST_AES_KeyHandle,
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes );

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_KG_AES",
                                  "CRYS_SST_TST_ATP_AES",
                                  TST_MasterTestName);


        /* Read the Key from the SST */
        /*---------------------------*/
        TST_Status = SST_AESKeyRead(TST_SessionId,
                                    TST_AES_KeyHandle,
                                    TST_OutputKey,
                                    &TST_SSTAESKeyType,
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AESKeyRead",
                                   "CRYS_SST_TST_ATP_AES",
                                    TST_MasterTestName);


        /*Encrypt the data using CRYS and extracted data(key) */
        /*----------------------------------------------------*/
        TST_Status = CRYS_AES(TST_IVCounter,
                              TST_OutputKey,
                              TST_OutputKeySizeBits,
                              CRYS_AES_Encrypt,
                              CRYS_AES_CBC_mode,
                              TST_InputData,
                              TST_VECTOR_SIZE,
                              TST_EncryptedData );

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES Encrypt",
                                   "CRYS_SST_TST_ATP_AES",
                                    TST_MasterTestName);


        /*Decrypt the data using CRYS_SST */
        /*--------------------------------*/
        TST_Status = CRYS_SST_AES(TST_AES_KeyHandle,
                                  TST_SessionId,
                                  TST_IVCounter,
                                  CRYS_AES_Decrypt,
                                  CRYS_AES_CBC_mode,
                                  TST_EncryptedData,
                                  TST_VECTOR_SIZE,
                                  TST_DecryptedData,
                                  TST_Workspace,
                                  TST_WorkspaceSizeInBytes); 
      
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_AES Decrypt",
                                   "CRYS_SST_TST_ATP_AES",
                                   TST_MasterTestName);

                                        
        /* Compare the decrypted result to the original */
        /*----------------------------------------------*/
        if(!DX_VOS_MemCmp(TST_DecryptedData, TST_InputData, TST_VECTOR_SIZE))
          TST_Status = TESTPASS; /*Data Buffers are the same*/
        else
          TST_Status = TESTFAIL; /*Data Buffers are different*/

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data",
                                   "CRYS_SST_TST_ATP_AES",
                                    TST_MasterTestName);

        
        /* Decrypt the data using the CRYS_SST non-integrated function */
        /*-------------------------------------------------------------*/
        TST_Status =  CRYS_SST_AES_Init(TST_AES_KeyHandle,
                                        TST_SessionId,
                                        &TST_AESCtx,
                                        TST_IVCounter,
                                        CRYS_AES_Decrypt,
                                        CRYS_AES_CBC_mode,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_AES_Init",
                                   "CRYS_SST_TST_ATP_AES",
                                   TST_MasterTestName);

        /* Call CRYS_AES_Block to decrypt the data */
        TST_Status = CRYS_AES_Block(&TST_AESCtx, TST_EncryptedData, TST_VECTOR_SIZE, TST_DecryptedData);
                       
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block Decrypt",
                                   "CRYS_SST_TST_ATP_AES",
                                   TST_MasterTestName);

        /* Call CRYS_AES_Free to release the context */
        TST_Status = CRYS_AES_Free(&TST_AESCtx);
        
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Free",
                                   "CRYS_SST_TST_ATP_AES",
                                 TST_MasterTestName);

        /* Compare the decrypted result to the original */
        /*----------------------------------------------*/
        if(!DX_VOS_MemCmp(TST_DecryptedData, TST_InputData, TST_VECTOR_SIZE))
          TST_Status = TESTPASS; /*Data Buffers are the same*/
        else
          TST_Status = TESTFAIL; /*Data Buffers are different*/

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data",
                                  "CRYS_SST_TST_ATP_AES",
                                   TST_MasterTestName);
 

        /* Delete the key */
        /*-----------------*/
        TST_Status = SST_DataDelete(TST_TransactionId,
                                    TST_SessionId,  
                                    TST_AES_KeyHandle,
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                                   "CRYS_SST_TST_ATP_AES",
                                   TST_MasterTestName);

        /*Delete the pass Authenticator*/
        TST_Status = SST_AuthenticatorDelete(TST_TransactionId,TST_PasswordAuthHandle);
       
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete",
                                   "CRYS_SST_TST_ATP_AES",
                                   TST_MasterTestName);


        END_OF_TEST:;
     }

     /* Close the transaction */
     /*-----------------------*/
     TST_Status = SST_TransactionEnd(TST_TransactionId,
                                     DX_TRUE,/*reflash sensitive*/
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes);

     TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                                "CRYS_SST_TST_ATP_AES",
                                TST_MasterTestName);


     /* Delete the session */
     /*--------------------*/
     TST_Status = SST_SessionDelete(TST_SessionId);

     TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                                "CRYS_SST_TST_ATP_AES",
                                TST_MasterTestName);

    }/*End of TST_HandleType "for loop" */
    return TST_Status;

EXIT_ON_ERROR:

    return DX_FAILURE;

}/* End of CRYS_SST_TST_ATP_AES */

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_SST_TST_ATP_DES
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 *    Test the connection between CRYS and the SSTfunctionality while using SST DES generated keys
 *    in the CRYS_SST module
 *   
 * Algorithm:       
 *    1.    Create a new session by calling: SST_SessionCreate.
 *    2.    Open a transaction by calling: SST_TransactionStart.
 *    3.    Create a new  password authenticator.
 *    4.    Close the transaction by calling: SST_TransactionEnd
 *    5.    Open the  password authentication by calling: SST_AuthenticationOpen.
 *    6.    Open a transaction by calling: SST_TransactionStart.
 *    7.    Generate DES keys inside the SST.
 *    8.    Read the Key from the SST by calling: SST_DESKeyRead.
 *    9.    Encrypt the data using CRYS and extracted data(key) by calling = CRYS_DES.
 *    10.   Decrypt the data using CRYS_SST by calling: CRYS_SST_DES.
 *    11.   Compare the decrypted result to the original+
 *    12.   Decrypt the data using CRYS_SST non-integrated function 
 *    13.   Compare the decrypted result to the original
 *    14.   Delete the DES key by calling: SST_DataDelete.
 *    15.   Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    16.   Close the transaction by calling: SST_TransactionClose.
 *    17.   Delete the session by calling: SST_SessionDelete.
 *                                
 *******************************************************************************/    
DxError_t CRYS_SST_TST_ATP_DES ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_SST_TST_ATP_DES";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_DES_KeyHandle,TST_InitialCtrlAuth,TST_PasswordAuthHandle;
    DxUint8_t                   TST_index;
    CRYS_DES_NumOfKeys_t        TST_CRYSDESNumOfKeys = CRYS_DES_1_KeyInUse;
    CRYS_DES_Key_t              TST_OutputKey;
    SSTDESKeyType_t             TST_SSTDESKeyType;
    CRYS_DESUserContext_t       TST_DESCtx;
    DxError_t                   TST_Status;

    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** CRYS_SST_TST_ATP_DES Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
   
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        if (TST_HandleType == TST_INTERNAL_HANDLE)
        {   
           ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
           ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
           ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
           TST_DES_KeyHandle.objId = SST_HANDLE_RANDOM_GENERATE;
         
        }
        else
        {
           ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
           ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
           ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
           TST_DES_KeyHandle.objId = TST_TEST_DEFINED_HENDLE;
        }


        /* Create a session for operations */
        /*---------------------------------*/
        TST_Status = SST_SessionCreate(&TST_SessionId);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


        /* Open a transaction for operations */
        /*-----------------------------------*/
        TST_Status = SST_TransactionStart(&TST_TransactionId);
      
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);

      

      /* Create a new  password authenticator */
      /*--------------------------------------*/
      TST_InitialCtrlAuth.objDigest = 0;
      TST_InitialCtrlAuth.objId = SST_CLOSE_AUTH_HANDLE;
      TST_PasswordAuthHandle.objDigest = 0;
      TST_PasswordAuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
      TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                   TST_SessionId, 
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterPassword,
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterPasswordSizeInBytes,
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterMaxNumOfTrials,
                                                   TST_InitialCtrlAuth,
                                                   &TST_PasswordAuthHandle,
                                                   TST_Workspace,
                                                   TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate ",
                                 CRYS_SST_TST_ATP_DES,
                                 TST_MasterTestName);


      /* Close the transaction */
      /*-----------------------*/
      TST_Status = SST_TransactionEnd(TST_TransactionId, DX_TRUE, TST_Workspace, TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                                 CRYS_SST_TST_ATP_DES,
                                 TST_MasterTestName);


      /* Open the  password authentication */
      /*-----------------------------------------*/
      TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                          TST_PasswordAuthHandle,
                                          TST_ATP_PassAuthObjectTestData[0].TST_MasterPassword,
                                          TST_ATP_PassAuthObjectTestData[0].TST_MasterPasswordSizeInBytes,
                                          TST_NO_CHALLENGE,
                                          TST_Workspace,
                                          TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen ",
                                 CRYS_SST_TST_ATP_DES,
                                  TST_MasterTestName);


      /* Open a transaction for operations */
      /*-----------------------------------*/
      TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                                 CRYS_SST_TST_ATP_AES,
                                 TST_MasterTestName);



        /* Setting test case specific parameters for key sizes */
        for (TST_index =0 ; TST_index < TST_NUM_OF_DES_KEYS ; TST_index++) /*To run on all key types vectors*/
        {
          switch(TST_index)
          {
            case 0:
               TST_CRYSDESNumOfKeys = CRYS_DES_1_KeyInUse;
                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** DES 1 Key\n"));
                break;
            case 1:
                goto END_OF_TEST; 
            case 2:
                TST_CRYSDESNumOfKeys = CRYS_DES_3_KeysInUse;
                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** DES 3 Keys\n"));
                break; 
            default:
                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n***  wrong DES Key \n"));
          };


          /* Generate DES keys inside the SST */
          /*----------------------------------*/       
          TST_Status = CRYS_SST_KG_DES(TST_TransactionId,
                                       TST_SessionId,
                                       TST_PasswordAuthHandle,
                                       TST_DES_KEY_TYPE,
                                       TST_CRYSDESNumOfKeys,
                                       &TST_DES_KeyHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_KG_DES", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /* Read the Key from the SST */
          /*---------------------------*/
          TST_Status = SST_DESKeyRead(TST_SessionId,
                                      TST_DES_KeyHandle,   
                                      (DxUint8_t *)&TST_OutputKey,
                                      &TST_SSTDESKeyType,
                                      TST_Workspace,
                                      TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DESKeyRead", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /*Encrypt the data using CRYS and extracted data(key) */
          /*----------------------------------------------------*/
          TST_Status = CRYS_DES(TST_IVCounter,
                               &TST_OutputKey,
                                TST_CRYSDESNumOfKeys,
                                CRYS_DES_Encrypt,
                                CRYS_DES_CBC_mode,
                                TST_InputData,
                                TST_VECTOR_SIZE,
                                TST_EncryptedData );

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_DES Encrypt", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /*Decrypt the data using CRYS_SST */
          /*--------------------------------*/
          TST_Status = CRYS_SST_DES(TST_DES_KeyHandle,
                                    TST_SessionId,
                                    TST_IVCounter,
                                    CRYS_DES_Decrypt,
                                    CRYS_DES_CBC_mode,
                                    TST_EncryptedData,
                                    TST_VECTOR_SIZE,
                                    TST_DecryptedData,
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);   

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_DES Decrypt", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /* Compare the decrypted result to the original */
          /*----------------------------------------------*/
          if(!DX_VOS_MemCmp(TST_DecryptedData, TST_InputData, TST_VECTOR_SIZE))
            TST_Status = TESTPASS; /*Data Buffers are the same*/
          else
            TST_Status = TESTFAIL; /*Data Buffers are different*/

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /*Decrypt the data using CRYS_SST non-integrated function */
          /*--------------------------------------------------------*/
          TST_Status = CRYS_SST_DES_Init(TST_DES_KeyHandle,
                                         TST_SessionId,
                                        &TST_DESCtx,
                                         TST_IVCounter,
                                         CRYS_DES_Decrypt,
                                         CRYS_DES_CBC_mode,
                                         TST_Workspace,
                                         TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_DES_Init", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /* Call CRYS_DES_Block to decrypt data */
          TST_Status = CRYS_DES_Block(&TST_DESCtx, TST_EncryptedData, TST_VECTOR_SIZE, TST_DecryptedData);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_DES_Block Decrypt", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /* Call CRYS_DES_Free to release the context*/
          TST_Status = CRYS_DES_Free(&TST_DESCtx);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_DES_Free", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


          /* Compare the decrypted result to the original */
          /*----------------------------------------------*/
          if(!DX_VOS_MemCmp(TST_DecryptedData, TST_InputData, TST_VECTOR_SIZE))
            TST_Status = TESTPASS; /*Data Buffers are the same*/
          else
          TST_Status = TESTFAIL; /*Data Buffers are different*/

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);
 

          /* Delete the key */
          /*-----------------*/
          TST_Status = SST_DataDelete(TST_TransactionId,
                                      TST_SessionId,  
                                      TST_DES_KeyHandle,
                                      TST_Workspace,
                                      TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);

        
          END_OF_TEST:;


        } /*End of key types loop*/


        /*Delete the pass Authenticator*/
        TST_Status = SST_AuthenticatorDelete(TST_TransactionId,TST_PasswordAuthHandle);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


        /* Close the transaction */
        /*-----------------------*/
        TST_Status = SST_TransactionEnd(TST_TransactionId,
                                        DX_TRUE,/*reflash sensitive*/
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);


        /* Delete the session */
        /*--------------------*/
        TST_Status = SST_SessionDelete(TST_SessionId);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete", "CRYS_SST_TST_ATP_DES", TST_MasterTestName);

    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
    return DX_FAILURE;

}/* End of CRYS_SST_TST_ATP_DES */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_SST_TST_ATP_HMAC
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 *    Test the connection between CRYS and the SSTfunctionality while using SST HMAC generated keys
 *    in the CRYS_SST module
 *       
 * Algorithm:       
 *    1.  Create a new session by calling: SST_SessionCreate.
 *    2.  Open a transaction by calling: SST_TransactionStart.
 *    3.  Create a new  password authenticator
 *    4.  Close the transaction
 *    5.  Open the  password authentication
 *    6.  Open a transaction for operations
 *    7.  Generate HMAC keys inside the SST.
 *    8.  Read the Key from the SST by calling: SST_HMACKeyRead.
 *    9.  HASH the data using CRYS and extracted data(key) by calling = CRYS_HMAC.
 *    10. HASH the data using CRYS_SST by calling: CRYS_SST_HMAC.
 *    11. Compare the HASH'd results to they need to be identical
 *    12. HASH the data using CRYS_SST non-integrated functions
 *    13. Call CRYS_HMAC_Update to calculate the HMAC
 *    14. Compare the HASH'd results to they need to be identical
 *    15. Delete the HMAC key by calling: SST_DataDelete.
 *    16. Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    17. Close the transaction by calling: SST_TransactionClose.
 *    18. Delete the session by calling: SST_SessionDelete.
 *                                
 *******************************************************************************/    
DxError_t CRYS_SST_TST_ATP_HMAC ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_SST_TST_ATP_HMAC";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_HMAC_KeyHandle,TST_InitialCtrlAuth,TST_PasswordAuthHandle;
    DxUint8_t                   TST_index;
    DxUint32_t                  TST_HMACKeySize;
    DxUint8_t                   TST_OutputKey[MAX_HAMC_KEY_SIZE];
    CRYS_HASH_Result_t          TST_CRYS_HASHResult,TST_CRYSSST_HASHResult;
    CRYS_HMACUserContext_t      TST_HMACCtx;
    DxError_t                   TST_Status;
    

    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** CRYS_SST_TST_ATP_HMAC Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;
   // TST_AuthHandle.objDigest = 0;
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_HMAC_KeyHandle.objId = SST_HANDLE_RANDOM_GENERATE;
               
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_HMAC_KeyHandle.objId = TST_TEST_DEFINED_HENDLE;
                
            }  


            /* Create a session for operations */
            /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          "CRYS_SST_TST_ATP_HMAC",
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          "CRYS_SST_TST_ATP_HMAC",
                          TST_MasterTestName);

       /* Create a new  password authenticator */
      /*--------------------------------------*/
      TST_InitialCtrlAuth.objDigest = 0;
      TST_InitialCtrlAuth.objId = SST_CLOSE_AUTH_HANDLE;
      TST_PasswordAuthHandle.objDigest = 0;
      TST_PasswordAuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
      TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                   TST_SessionId, 
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterPassword,
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterPasswordSizeInBytes,
                                                   TST_ATP_PassAuthObjectTestData[0].TST_MasterMaxNumOfTrials,
                                                   TST_InitialCtrlAuth,
                                                   &TST_PasswordAuthHandle,
                                                   TST_Workspace,
                                                   TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate ",
                                 CRYS_SST_TST_ATP_HMAC,
                                 TST_MasterTestName);


      /* Close the transaction */
      /*-----------------------*/
      TST_Status = SST_TransactionEnd(TST_TransactionId, DX_TRUE, TST_Workspace, TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                                 CRYS_SST_TST_ATP_HMAC,
                                 TST_MasterTestName);


      /* Open the  password authentication */
      /*-----------------------------------------*/
      TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                          TST_PasswordAuthHandle,
                                          TST_ATP_PassAuthObjectTestData[0].TST_MasterPassword,
                                          TST_ATP_PassAuthObjectTestData[0].TST_MasterPasswordSizeInBytes,
                                          TST_NO_CHALLENGE,
                                          TST_Workspace,
                                          TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen ",
                                 CRYS_SST_TST_ATP_HMAC,
                                  TST_MasterTestName);


      /* Open a transaction for operations */
      /*-----------------------------------*/
      TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                                 CRYS_SST_TST_ATP_HMAC,
                                 TST_MasterTestName);


            /* Setting test case specific parameters for key sizes */
            for (TST_index =0 ; TST_index < TST_NUM_OF_HMAC_KEYS ; TST_index++) /*To run on all key types vectors*/
            {
                switch(TST_index)
            {
                        case 0:
                                TST_HMACKeySize = TST_HMAC_4BYTE_KEY;
                                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** HMAC Key 4Byts\n"));
                                break;
                        case 1:
                                TST_HMACKeySize = TST_HMAC_32BYTE_KEY;
                                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** HMAC Key 32Byts\n"));
                                break; 
                        case 2:
                                TST_HMACKeySize = TST_HMAC_65BYTE_KEY;
                                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n*** HMAC Key 65Byts\n"));
                                break; 
                        default:
                                ATP_LOG_TST_PRINT((SST_TST_MSG,"\n***  wrong HAMC Key size \n"));

             };

                /* Generate HMAC keys inside the SST */
                /*-----------------------------------*/     
                TST_Status = CRYS_SST_KG_HMAC(TST_TransactionId,
                                             TST_SessionId,
                                             TST_PasswordAuthHandle,
                                             TST_HMAC_KEY_TYPE,
                                             TST_HMACKeySize,
                                             &TST_HMAC_KeyHandle,
                                             TST_Workspace,
                                             TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_KG_HMAC",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);


                /* Read the Key from the SST */
                /*---------------------------*/
                TST_Status = SST_HMACKeyRead(TST_SessionId,
                                           TST_HMAC_KeyHandle,   
                                           TST_OutputKey,
                                           &TST_HMACKeySize,
                                           TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_HMACKeyRead",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);


                /*HASH the data using CRYS and extracted data(key) */
                /*-------------------------------------------------*/
                TST_Status = CRYS_HMAC(CRYS_HASH_SHA1_mode,
                                      TST_OutputKey,
                                      (DxUint16_t)TST_HMACKeySize,
                                      TST_InputData,
                                      sizeof(TST_InputData),
                                      TST_CRYS_HASHResult);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_HMAC",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);


                /*HASH the data using CRYS_SST */
                /*-----------------------------*/
                TST_Status = CRYS_SST_HMAC(TST_HMAC_KeyHandle,
                        TST_SessionId,
                        CRYS_HASH_SHA1_mode,
                        TST_InputData,
                        sizeof(TST_InputData),
                        TST_CRYSSST_HASHResult,
                                          TST_Workspace,
                                          TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_HMAC",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);


                /* Compare the decrypted result to the original */
                /*----------------------------------------------*/
             if(!DX_VOS_MemCmp(TST_CRYSSST_HASHResult,
                    TST_CRYS_HASHResult,
                    sizeof(CRYS_HASH_Result_t)))
                                  TST_Status = TESTPASS; /*Data Buffers are the same*/
                                  else
                                  TST_Status = TESTFAIL; /*Data Buffers are different*/

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - HASH results",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);
 
          
          /*HASH the data using CRYS_SST non-integrated functions */
                /*------------------------------------------------------*/
          TST_Status = CRYS_SST_HMAC_Init(TST_HMAC_KeyHandle,
                         TST_SessionId,
                         &TST_HMACCtx,
                         CRYS_HASH_SHA1_mode,
                         TST_Workspace,
                                               TST_WorkspaceSizeInBytes);
          
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_SST_HMAC_Init",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);

          
          /* Call CRYS_HMAC_Update to calculate the HMAC*/
          TST_Status = CRYS_HMAC_Update(&TST_HMACCtx, TST_InputData,
                         sizeof(TST_InputData));
          
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_HMAC_Update",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);
  
          
          TST_Status = CRYS_HMAC_Finish(&TST_HMACCtx, TST_CRYSSST_HASHResult);
                       
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_HMAC_Finish",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);
  
                       
          /* Compare the decrypted result to the original */
                /*----------------------------------------------*/
             if(!DX_VOS_MemCmp(TST_CRYSSST_HASHResult,
                    TST_CRYS_HASHResult,
                    sizeof(CRYS_HASH_Result_t)))
                                  TST_Status = TESTPASS; /*Data Buffers are the same*/
                                  else
                                  TST_Status = TESTFAIL; /*Data Buffers are different*/

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - HASH results",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);


                /* Delete the key */
                /*-----------------*/
                 TST_Status = SST_DataDelete(TST_TransactionId,
                                            TST_SessionId,  
                                            TST_HMAC_KeyHandle,
                                            TST_Workspace,
                                            TST_WorkspaceSizeInBytes);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                              "CRYS_SST_TST_ATP_HMAC",
                              TST_MasterTestName);

            } /*End of key types loop*/


            
        /*Delete the pass Authenticator*/
        TST_Status = SST_AuthenticatorDelete(TST_TransactionId,TST_PasswordAuthHandle);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete", "CRYS_SST_TST_ATP_HMAC", TST_MasterTestName);


       
            /* Close the transaction */
            /*-----------------------*/
      TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          "CRYS_SST_TST_ATP_HMAC",
                          TST_MasterTestName);


            /* Delete the session */
            /*--------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                          "CRYS_SST_TST_ATP_HMAC",
                          TST_MasterTestName);

    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of CRYS_SST_TST_ATP_HMAC */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
#endif

